<?php
declare(strict_types=1);

// =============================
// Developer Studio Live Code - Responsivo
// =============================

function sanitize_input(string $data): string {
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

function real_ip(): string {
    $ip = $_SERVER['REMOTE_ADDR'] ?? getenv('REMOTE_ADDR');
    foreach (['HTTP_X_FORWARDED_FOR','HTTP_CLIENT_IP'] as $h) {
        if (!empty($_SERVER[$h]) && filter_var($_SERVER[$h], FILTER_VALIDATE_IP)) {
            $ip = $_SERVER[$h];
            break;
        }
    }
    return sanitize_input($ip);
}

function get_os(): string {
    $ua = strtolower($_SERVER["HTTP_USER_AGENT"] ?? '');
    $os_array = [
        '/windows nt 10/i'=>'Windows 10','/windows nt 6.3/i'=>'Windows 8.1',
        '/windows nt 6.2/i'=>'Windows 8','/windows nt 6.1/i'=>'Windows 7',
        '/macintosh|mac os x/i'=>'Mac OS X','/linux/i'=>'Linux',
        '/android/i'=>'Android','/iphone/i'=>'iPhone'
    ];
    foreach($os_array as $regex=>$value) if(preg_match($regex,$ua)) return $value;
    return 'Unknown OS';
}

function browser_type(): string {
    $ua = strtolower($_SERVER["HTTP_USER_AGENT"] ?? '');
    $browsers = [
        '/msie|trident/i'=>'Internet Explorer','/firefox/i'=>'Firefox',
        '/safari/i'=>'Safari','/chrome/i'=>'Chrome','/edge/i'=>'Edge','/opera/i'=>'Opera'
    ];
    foreach($browsers as $regex=>$value) if(preg_match($regex,$ua)) return $value;
    return 'Unknown Browser';
}

function get_device(): string {
    $ua = strtolower($_SERVER["HTTP_USER_AGENT"] ?? '');
    if(preg_match("/(tablet|ipad|playbook)|(android(?!.*mobi))/i",$ua)) return 'Tablet';
    if(preg_match("/(mobi|iphone|ipod|blackberry|phone)/i",$ua)) return 'Mobile';
    return 'Computer';
}

function is_tor_exit_point(): string {
    $remote = $_SERVER["REMOTE_ADDR"] ?? '';
    $server = $_SERVER["SERVER_ADDR"] ?? '';
    $port   = $_SERVER["SERVER_PORT"] ?? '';
    $reverse = implode('.', array_reverse(explode('.', $remote))) . "." . $port . "." . implode('.', array_reverse(explode('.', $server))) . ".ip-port.exitlist.torproject.org";
    return (gethostbyname($reverse)==="127.0.0.2")?'True':'False';
}

// Captura IP e informações
$ipl = real_ip();
$ch = curl_init("https://ipinfo.io/{$ipl}/json");
curl_setopt($ch, CURLOPT_RETURNTRANSFER,true);
curl_setopt($ch, CURLOPT_TIMEOUT,3);
$response = curl_exec($ch);
curl_close($ch);
$details = @json_decode($response);
$country = sanitize_input($details->country ?? 'Unknown');
$state   = sanitize_input($details->region ?? 'Unknown');
$city    = sanitize_input($details->city ?? 'Unknown');
$isp     = isset($details->org)?sanitize_input(preg_replace("/AS\\d+\s/","",$details->org)):'Unknown';
$loc     = sanitize_input($details->loc ?? 'Unknown');

date_default_timezone_set("Europe/London");

$log_line = "---------------------------------------------\n"
    . "[TOA] ".date("d-m-Y H:i:s")."\n"
    . "[IPV6] {$ipl}\n"
    . "[Country] {$country}\n"
    . "[City] {$city}\n"
    . "[State] {$state}\n"
    . "[ISP] {$isp}\n"
    . "[Location] {$loc}\n"
    . "[UA] ".sanitize_input($_SERVER["HTTP_USER_AGENT"] ?? '')."\n"
    . "[OS] ".get_os()."\n"
    . "[Browser] ".browser_type()."\n"
    . "[Device] ".get_device()."\n"
    . "[Tor Browser] ".is_tor_exit_point()."\n";

@file_put_contents("snoop/".date("d-m-Y H-i-s").".log", $log_line.PHP_EOL, FILE_APPEND);
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Enter Panel - Alert!</title>
<meta name="viewport" content="width=device-width, initial-scale=1, maximum-scale=1">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Share+Tech+Mono&family=Montserrat:wght@700&display=swap" rel="stylesheet">
<style>
/* Reset e base */
* { box-sizing: border-box; margin:0; padding:0; }
body {
    height: 100vh;
    display: flex;
    justify-content: center;
    align-items: center;
    font-family: 'Share Tech Mono', monospace;
    background: linear-gradient(-45deg, #1a1a1a, #0f2027, #203a43, #2c5364);
    background-size: 400% 400%;
    animation: gradientBG 15s ease infinite;
    color: #fff;
}
@keyframes gradientBG {
    0% { background-position:0% 50%; }
    50% { background-position:100% 50%; }
    100% { background-position:0% 50%; }
}
.container {
    background: rgba(0,0,0,0.55);
    backdrop-filter: blur(10px);
    border-radius: 1.2em;
    padding: 2.5em;
    text-align: center;
    max-width: 600px;
    width: 90%;
    box-shadow: 0 0 30px rgba(0,255,200,0.5);
    transition: transform 0.3s ease;
}
.container:hover {
    transform: scale(1.02);
}
.container img {
    width: 35%;
    max-width: 200px;
    height: auto;
    margin-bottom: 20px;
}
.container p {
    margin: 10px 0;
    font-size: 1rem;
}
.btn {
    display: inline-block;
    padding: 12px 28px;
    background: #28a745;
    border-radius: 10px;
    color: #fff;
    font-weight: bold;
    text-decoration: none;
    transition: all 0.3s ease;
}
.btn:hover {
    background: #36d65a;
    transform: scale(1.05);
    box-shadow: 0 0 12px #36d65a;
}

/* Responsividade */
@media (max-width:768px) {
    .container { padding:2em; }
    .container p { font-size: 0.9rem; }
    .btn { padding:10px 22px; font-size:0.9rem; }
}
@media (max-width:480px) {
    .container { padding:1.5em; }
    .container img { width: 50%; }
    .container p { font-size:0.85rem; }
    .btn { padding:8px 20px; font-size:0.85rem; }
}
</style>
</head>
<body>
<div class="container">
    <img src="img/logo.png" alt="Logo">
    <p><strong>Welcome!</strong></p>
    <p>This is Manager Panel.</p>
    <p>Please change your User/Pass from admin to a safer one.</p>
    <p>Click Continue to proceed.</p>
    <a class="btn" href="./profile.php">✔ Continue</a>
</div>
</body>
</html>
